<?php

namespace Nette\Admin;

use function Composer\Autoload\includeFile;
use File;
use Route;
use Config;

class ModuleLoader
{
    public function __construct()
    {
        $this->vendorPath = base_path() . '/vendor';
        $this->localPath = base_path() . '/app/Packages';
        $this->dbMigrationPath = base_path() . '/database/migrations';
        $this->dbSeedPath = base_path() . '/database/seeds';
    }

    public $menus;
    public $permissions;
    public $routes;

    public $vendorPath;
    public $localPath;
    public $dbMigrationPath;
    public $dbSeedPath;

    public $loadedModules = [];

    public $composerModuleList = [];
    public $localModuleList = [];


    public function load()
    {
        //TODO: cache when prod

        //First Load Composer Packages
        $composerPackages = [];
        if (File::exists($path = $this->vendorPath . '/composer/installed.json')) {
            $composerPackages = json_decode(File::get($path), true);
        }

        $composerPackages = collect($composerPackages)->mapWithKeys(function ($package) {
            return [$this->extractPackageName($package['name']) => $package['extra']['nette']['module'] ?? []];
        })->filter()->all();

        $this->composerModuleList = array_keys($composerPackages);

        $this->loadComposerModules();

        if (File::exists($this->localPath)) {
            $this->localModuleList = array_map('basename', File::directories($this->localPath));
            //TODO: check module structure

            $this->loadLocalModules();
        }


        //Second Load Local Packages;
    }


    protected function prepareModule($moduleName, $type = 'local')
    {
        $this->loadedModules[$moduleName] = ['type' => $type];
    }

    protected function loadModuleInfo($moduleName, $modulePath)
    {
        $moduleInfo = null;
        $filePath = $modulePath . '/module_info.php';
        if (File::exists($filePath)) {
            $moduleInfo = include $filePath;
        }
        $this->loadedModules[$moduleName]['info'] = $moduleInfo;

        view()->addNamespace($moduleName, $modulePath . '/views');

        $helperPath = $modulePath . '/helpers.php';
        if (File::exists($helperPath)) {
            include $helperPath;
        }

    }

    protected function loadModuleMenu($moduleName, $modulePath)
    {
        $moduleInfo = null;
        $moduleMenuData = [];
        $filePath = $modulePath . '/module_menus.php';
        if (File::exists($filePath)) {
            $moduleMenuData = include $filePath;
        }
        $this->loadedModules[$moduleName]['menu'] = $moduleMenuData;
    }

    protected function loadModuleNavs($moduleName, $modulePath)
    {

        $moduleInfo = null;
        $moduleNavData = [];
        $filePath = $modulePath . '/module_navs.php';
        if (File::exists($filePath)) {
            $moduleNavData = include $filePath;
        }
        $this->loadedModules[$moduleName]['nav'] = $moduleNavData;
    }

    protected function loadModuleCaches($moduleName, $modulePath)
    {

        $moduleInfo = null;
        $moduleCacheData = [];
        $filePath = $modulePath . '/module_caches.php';
        if (File::exists($filePath)) {
            $moduleCacheData = include $filePath;
        }
        $this->loadedModules[$moduleName]['cache'] = $moduleCacheData;
    }

    protected function loadModuleRoutes($moduleName, $modulePath)
    {
        $moduleInfo = null;
        $filePath = $modulePath . '/module_routes.php';
        if (File::exists($filePath)) {
            $this->registerRoutes($filePath);
        }


    }

    protected function loadModuleDBEntities($moduleName, $modulePath)
    {
        $moduleInfo = null;
        $moduleNavData = [];
        $migrationPath =    $modulePath . '/migrations';
        $seedPath =       $modulePath . '/seeds';
        if (File::exists($migrationPath)) {
            $this->loadedModules[$moduleName]['migrations'] = File::glob($migrationPath.DIRECTORY_SEPARATOR.'*.php');
        }

        if (File::exists($seedPath)) {
            $this->loadedModules[$moduleName]['seeds'] = File::glob($seedPath.DIRECTORY_SEPARATOR.'*.php');
        }


    }

    private function registerRoutes($routeFile)
    {
        Route::group(['prefix' => config('admin.prefix'), 'as' => 'admin.', 'middleware' => ['web']], function () use ($routeFile) {
            Route::group(['middleware' => [\Nette\Admin\Middlewares\RedirectIfNotAdmin::class]], function () use ($routeFile) {
                include $routeFile;
            });
        });
    }

    protected function loadModuleSettings($moduleName, $modulePath)
    {
        $moduleSettings = null;
        $filePath = $modulePath . '/module_settings.php';
        if (File::exists($filePath)) {
            $loadSettings = $this->loadedModules[$moduleName]['info']['settings_key'] ?? false;
            if ($loadSettings) {
                $settingsKey = $this->loadedModules[$moduleName]['info']['settings_key'];
                $settingsName = $this->loadedModules[$moduleName]['info']['settings_name'];
                $settings = config('settings', []);
                $settings['tabs'][$settingsKey] = $settingsName;


                $loadedSettings = include $filePath;

                array_walk($loadedSettings, function (&$item) use ($settingsKey) {
                    $item['tab'] = $settingsKey;
                });

                $settings['fields'] = array_merge($settings['fields'] ?? [], $loadedSettings);

                Config::set('settings', $settings);

            };


        }
    }

    protected function loadModulePermissions($moduleName, $modulePath)
    {
        $modulePermissions = null;
        $filePath = $modulePath . '/module_permissions.php';
        if (File::exists($filePath)) {
            $loadedPermissions = include $filePath;
            $this->permissions[$moduleName] = $loadedPermissions;
            $this->loadedModules[$moduleName]['permissions'] = $loadedPermissions;
        }


    }

    protected function loadModuleUploads($moduleName, $modulePath)
    {
        $moduleSettings = null;
        $filePath = $modulePath . '/module_upload.php';
        if (File::exists($filePath)) {

            $loadedUploads = include $filePath;

            $currentUploads = config('upload', []);

            $imgUploads = $currentUploads['img']['keys'] ?? [];
            $currentUploads['img']['keys'] = array_merge($imgUploads, ($loadedUploads['img'] ?? []));


            $docUploads = $currentUploads['doc']['keys'] ?? [];
            $currentUploads['doc']['keys'] = array_merge($docUploads, ($loadedUploads['doc'] ?? []));

            Config::set('upload', $currentUploads);
        }
    }

    //region Composer Spesific Loading
    public function loadComposerModules()
    {
        foreach ($this->composerModuleList as $composerModule) {
            $this->loadComposerModule($composerModule);
        }
    }

    public function loadComposerModule($moduleName)
    {
        $modulePath = $this->composerModulePath($moduleName);
        if (File::exists($modulePath)) {
            $this->prepareModule($moduleName, 'composer');
            $this->loadModuleInfo($moduleName, $modulePath);
            $this->loadModuleMenu($moduleName, $modulePath);
            $this->loadModuleMenu($moduleName, $modulePath);
            $this->loadModuleNavs($moduleName, $modulePath);
            $this->loadModuleRoutes($moduleName, $modulePath);
            $this->loadModuleSettings($moduleName, $modulePath);
            $this->loadModulePermissions($moduleName, $modulePath);
            $this->loadModuleUploads($moduleName, $modulePath);
            $this->loadModuleDBEntities($moduleName, $modulePath);
            $this->loadModuleCaches($moduleName, $modulePath);

        }
    }

    private function composerModulePath($packagename = '')
    {
        return $this->vendorPath . '/' . $packagename;
    }

    private function extractPackageName($package)
    {
        return str_replace($this->vendorPath . '/', '', $package);
    }

    //endregion

    //region Local Spesific Loading
    public function loadLocalModules()
    {
        foreach ($this->localModuleList as $localModule) {
            $this->loadLocalModule($localModule);
        }
    }

    public function loadLocalModule($moduleName)
    {
        $modulePath = $this->localModulePath($moduleName);
        if (File::exists($modulePath)) {
            $this->prepareModule($moduleName, 'local');
            $this->loadModuleInfo($moduleName, $modulePath);
            $this->loadModuleMenu($moduleName, $modulePath);
            $this->loadModuleNavs($moduleName, $modulePath);
            $this->loadModuleRoutes($moduleName, $modulePath);
            $this->loadModuleSettings($moduleName, $modulePath);
            $this->loadModulePermissions($moduleName, $modulePath);
            $this->loadModuleUploads($moduleName, $modulePath);
            $this->loadModuleDBEntities($moduleName, $modulePath);
            $this->loadModuleCaches($moduleName, $modulePath);
        }
    }

    private function localModulePath($packagename = '')
    {
        return $this->localPath . '/' . $packagename;
    }
    //endregion

    public function generateMenu($active = false,$horizontal = false)
    {
        $menuData = [];

        $menuGroups = [];
        $menuItems = [];

        foreach ($this->loadedModules as $module) {
            if ($module['menu'] ?? false) {
                if ($module['menu']['groups'] ?? false) {
                    foreach ($module['menu']['groups'] as $key => $moduleMenuEntry) {
                        $menuGroups[] = $moduleMenuEntry + ['key' => $key];
                    }
                }

                if ($module['menu']['non-groups'] ?? false) {
                    foreach ($module['menu']['non-groups'] as $key => $moduleMenuEntry) {
                        $menuItems[] = $moduleMenuEntry + ['key' => $key];
                    }
                }
            }
        }

        usort($menuGroups, function ($a, $b) {
            return ($a['order'] ?? 0) <=> ($b['order'] ?? 0);
        });
        usort($menuItems, function ($a, $b) {
            return ($a['order'] ?? 0) <=> ($b['order'] ?? 0);
        });


        foreach ($menuItems as $menuItemKey => $menuItem) {
            if ((!isset($menuItem['permission'])) || (isset($menuItem['permission']) && checkPermission($menuItem['permission']))) {
                $menuData[0]['name'] = '';
                $menuData[0]['items'][$menuItemKey] = $menuItem;
            }
        }


        foreach ($menuGroups as $menuGroup) {
            foreach ($menuGroup['items'] ?? [] as $menuGroupItemKey => $menuGroupItem) {
                if ((!isset($menuGroupItem['permission'])) || (isset($menuGroupItem['permission']) && checkPermission($menuGroupItem['permission']))) {
                    $menuData[$menuGroup['key']]['name'] = $menuGroup['name'];
                    $menuData[$menuGroup['key']]['icon'] = $menuGroup['icon'] ?? null;
                    $menuData[$menuGroup['key']]['items'][$menuGroupItemKey] = $menuGroupItem + ['key' => $menuGroupItemKey];
                }
            }

        }

        $view = $horizontal ? 'admin::_partials.horizontalmenu' : 'admin::_partials.menu';
        $menu = view($view)->with('menuData', $menuData)->with('activeMenu', $active)->render();

        return PHP_EOL . $menu;
    }

    public function generateNav()
    {
        $navData = [];

        $navItems = [];



        foreach ($this->loadedModules as $module) {
            if ($module['nav'] ?? false) {
                foreach ($module['nav'] as $key => $moduleNavEntry) {
                    $navItems[] = $moduleNavEntry + ['key' => $key];
                }
            }
        }
        usort($navItems, function ($a, $b) {
            return ($a['order'] ?? 0) <=> ($b['order'] ?? 0);
        });



        foreach ($navItems as $navItem) {
                if ((!isset($navItem['permission'])) || (isset($navItem['permission']) && checkPermission($navItem['permission']))) {
                    $navData[$navItem['key']] = $navItem;
                }
        }
        $navbarRender = view('admin::_partials.navbar-generate')->with('navData', $navData)->render();

        return PHP_EOL . $navbarRender;
    }

    public function getModuleData()
    {
        return $this->loadedModules;
    }

    public function getPermissionSet()
    {
        return $this->permissions;
    }

    public function getDBMigrationPath()
    {
        return $this->dbMigrationPath;
    }

    public function getDBSeedPath()
    {
        return $this->dbSeedPath;
    }



}
