<?php

namespace Nette\Admin;

use Carbon\Carbon;
use File;
use Route;
use Config;

class FilterGenerator
{

    public $filters;
    public $view;
    public $route;

    public $columnPrefix = 'f_';

    public $currentRequest;

    private function getFilterParameter($column)
    {
        return $this->columnPrefix . $column;
    }

    private function getFilterColumn($parameter)
    {
        return ltrim($parameter, $this->columnPrefix);
    }

    private function getFilterData($column)
    {
        return $this->currentRequest->get($this->getFilterParameter($column), '');
    }

    private function getFilterChainData($column, $filter)
    {
        $chainRootID = $this->currentRequest->get($this->getFilterParameter($column), '');
        return $filter['relation_class']::{$filter['relation_getvaluemethod']}($chainRootID);
    }

    private function isFilterDataValid($column, $filter)
    {
        if($filter['type'] == 'range'){
            $validData = (array)$this->getFilterData($column);
            return (isset($validData['min']) || isset($validData['max']));
        }

        $validData = $this->getFilterData($column);

        return isset($validData) && $validData != '';
    }

    public function __construct()
    {
        $this->currentRequest = request();
    }

    private function like($builder, $column, $start = true, $end = false)
    {
        return $builder->where($column, 'LIKE', (($start ? '%' : '') . ($this->getFilterData($column)) . ($end ? '%' : '')));
    }

    private function equal($builder, $column)
    {
        return $builder->where($column, '=', $this->getFilterData($column));
    }

    private function range($builder, $column)
    {
        $rangeValue = (array)$this->getFilterData($column);

        $min = $max = null;

        if (!empty($rangeValue['min'])) $min = trim($rangeValue['min']);
        if (!empty($rangeValue['max'])) $max = trim($rangeValue['max']);

        if (isset($min) || isset($max)) {
            if (isset($min) && !isset($max)) { // Sadece MIN varsa
                $builder = $builder->where($column, '>', $rangeValue['min']);
            } else if (!isset($min) && isset($max)) { // Sadece MAX varsa
                $builder = $builder->where($column, '<', $rangeValue['max']);
            } else { // Her ikisi de varsa
                $builder = $builder->whereBetween($column, $rangeValue);
            }
        }

        return $builder;
    }

    private function daterange($builder,$column)
    {
        $rangeValue = $this->getFilterData($column);

        $dates = explode(' - ',$rangeValue);

        $start = $end = null;

        if(count($dates) ==2){
            $start = $dates[0];
            $end = $dates[1];
        } else {
            return $builder;
        }

        $start = Carbon::createFromFormat('d/m/Y', $start)->startOfDay();
        $end = Carbon::createFromFormat('d/m/Y', $end)->endOfDay();

        $builder = $builder->whereBetween($column,[$start,$end]);

        return $builder;
    }

    private function startdate($builder,$column)
    {
        $dateValue = $this->getFilterData($column);
        $start = Carbon::createFromFormat('d/m/Y', $dateValue)->startOfDay();
        $builder = $builder->where($column,'>',$start);
        return $builder;
    }

    private function enddate($builder,$column)
    {
        $dateValue = $this->getFilterData($column);
        $end = Carbon::createFromFormat('d/m/Y', $dateValue)->endOfDay();
        $builder = $builder->where($column,'<',$end);
        return $builder;
    }

    private function date($builder,$column)
    {
        $dateValue = $this->getFilterData($column);
        $start = Carbon::createFromFormat('d/m/Y', $dateValue)->startOfDay();
        $end = Carbon::createFromFormat('d/m/Y', $dateValue)->endOfDay();
        $builder = $builder->whereBetween($column,[$start,$end]);
        return $builder;
    }

    private function in($builder, $column, $filter = null, $chain = false)
    {

        if ($chain) {
            $filterData = $this->getFilterChainData($column, $filter);
        } else {
            $filterData = $this->getFilterData($column);

            if (!is_array($filterData)) {
                $filterData = [$filterData];
            }
        }

        if (count($filterData) < 1) {
            return $builder;
        }

        return $builder->whereIn($column, $filterData);
    }

    public function apply($filters = null, $builder)
    {
        $activeFilters = [];
        if (isset($filters)) {
            $this->filters = $filters;
        }

        $eloquentBuilder = $builder;
        $request = request();

        $filterApplied = ($request->get('apply_filter', '') == 'true');

        $isQuickSearch = false;
        if ($filterApplied) {
            //qs means quick search
            $isQuickSearch = ($request->get('qs', '') == '');
        }


        //Boş olmayan Request itemları ile Bütün filtrelerin kesişim kümesi seçilmiş filtreleri belirliyor.
        $selectedFilters = array_intersect_key(
            $this->filters, array_flip(
            array_map(
                function ($item) {
                    return $this->getFilterColumn($item);
                },
                array_keys(
                    array_filter(
                        $request->all(),
                        function ($val) {
                            return $val != '';
                        }
                    )
                )
            )
        ));

        if ($filterApplied) {
            if (count($selectedFilters) > 0) {
                if ($isQuickSearch) {
                    //Not A quicksearch
                    foreach ($selectedFilters as $filterKey => $filter) {
                        if ($this->isFilterDataValid($filterKey, $filter)) {

                            $isChain = false;
                            if (in_array($filter['type'], ['chain_relation', 'multi_chain_relation'])) {
                                $isChain = true;
                            }

                            switch ($filter['operator'] ?? null) {
                                case 'like':
                                    $eloquentBuilder = $this->like($eloquentBuilder, $filterKey, true, true);
                                    break;
                                case 'like_start':
                                    $eloquentBuilder = $this->like($eloquentBuilder, $filterKey, false, true);
                                    break;
                                case 'like_end':
                                    $eloquentBuilder = $this->like($eloquentBuilder, $filterKey, true, false);
                                    break;
                                case 'equal':
                                    $eloquentBuilder = $this->equal($eloquentBuilder, $filterKey);
                                    break;
                                case 'in':
                                    $eloquentBuilder = $this->in($eloquentBuilder, $filterKey, $filter, $isChain);
                                    break;
                                case 'range':
                                    $eloquentBuilder = $this->range($eloquentBuilder, $filterKey);
                                    break;
                                case 'daterange':
                                    $eloquentBuilder = $this->daterange($eloquentBuilder, $filterKey);
                                    break;
                                case 'startdate':
                                    $eloquentBuilder = $this->startdate($eloquentBuilder, $filterKey);
                                    break;
                                case 'enddate':
                                    $eloquentBuilder = $this->enddate($eloquentBuilder, $filterKey);
                                    break;
                                case 'date':
                                    $eloquentBuilder = $this->date($eloquentBuilder, $filterKey);
                                    break;
                                default:
                                    break;
                            }
                            $activeFilters[$filterKey] = $this->getFilterData($filterKey);
                        }
                    }

                } else {
                    $quickSearchTerm = $request->get("qs", '');
                    if (is_numeric($quickSearchTerm)) {
                        //ID Search
                        $eloquentBuilder = $eloquentBuilder->where('id', '=', $quickSearchTerm);
                    } elseif (mb_strlen($quickSearchTerm) > 0) {
                        //Name Search
                        $eloquentBuilder = $eloquentBuilder->where('name', 'LIKE', '%' . $quickSearchTerm . '%');
                    }
                }
            }
        }


        view()->share('filtered', true);
        view()->share('activeFilters', $activeFilters);

        return $eloquentBuilder;
    }


    public function button()
    {
        return view('admin::_partials.filters.button');
    }

    public function render()
    {
        $rawFilters = view()->shared('allFilters');
        $allFilters = [];

        foreach ($rawFilters as $rawFilterKey => $rawFilter) {
            $allFilters[$rawFilterKey] = $rawFilter;
            $allFilters[$rawFilterKey]['filter_parameter'] = $this->getFilterParameter($rawFilterKey);
        }

        $activeFilters = (view()->shared('activeFilters')) ?? [];
        //dd($activeFilters);
        return view('admin::_partials.filters.index', ['allFilters' => $allFilters, 'activeFilters' => $activeFilters]);
    }

    //region URL HELPERS
    public function removeFilterItem($filter)
    {
        $removingFilter = $this->getFilterParameter($filter);

        $data = request()->all();

        $data = array_filter($data, function ($var) {
            return !is_null($var) && $var != '';
        });

        //always remove page
        unset($data['page']);
        unset($data[$removingFilter]);
        //if there is no filter remove apply_filter
        if (count($data) == 1) unset($data["apply_filter"]);

        return url()->current() . '?' . http_build_query($data);

    }

    public function getFilterBaseUrl()
    {
        return url()->current();
    }


    public function getRelationProperty($filter, $value)
    {
        try {
            if (is_array($value)) {
                return implode(' , ', $filter['relation_class']::whereIn('id', $value)->get()->pluck($filter['relation_field'])->toArray());
            } else {
                return $filter['relation_class']::find($value)->{$filter['relation_field']};
            }
        } catch (\Exception $ex) {
            return 'Tanımsız';
        }
    }

    //endregion

    public function getFilterPresentation($allFilters, $activeFilter, $activeFilterKey)
    {
        $return = 'Tanımsız';
        if (in_array($allFilters[$activeFilterKey]['type'], ['relation', 'multi_relation', 'chain_relation', 'multi_chain_relation'])) {
            $return = FilterGenerator::getRelationProperty($allFilters[$activeFilterKey], $activeFilter);
        } else if ($allFilters[$activeFilterKey]['type'] == 'range') {

            if (isset($activeFilter['max']) && isset($activeFilter['min'])) {
                $return = $activeFilter['min'] . ' <-> ' . $activeFilter['max'];
            } else if (isset($activeFilter['max'])) {
                $return = '<' . $activeFilter['max'];
            } else if (isset($activeFilter['min'])) {
                $return = $activeFilter['min'] . '>';
            }
        } else {
            if (isset($allFilters[$activeFilterKey]['values'])) {
                if (is_array($activeFilter) && count($activeFilter) > 0) {
                    //dd(array_values($allFilters[$activeFilterKey]['values']));
                    $return = implode(' , ', array_intersect_key($allFilters[$activeFilterKey]['values'], array_flip(array_values($activeFilter))));

                } else {
                    $return = $allFilters[$activeFilterKey]['values'][$activeFilter] ?? 'Tanımsız';
                }
            } else {
                $return = $activeFilter;
            }
        }


        return $return;
    }

}
