<?php

namespace App\Http\Controllers\Front;

use App\Http\Controllers\FrontBaseController;
use App\Http\Requests\ContactRequest;
use App\Packages\Article\Models\Article;
use App\Packages\Category\Models\Category;
use App\Packages\Faq\Models\Faq;
use App\Packages\Newsletter\Models\Newsletter;
use App\Packages\Newsletter\Requests\NewsletterRequest;
use App\Packages\Page\Models\Page;
use App\Packages\Product\Models\Product;
use App\Packages\Redirect\Models\Redirect;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Nette\Contact\Models\Contact;
use Illuminate\Database\Eloquent\Collection;

class FormController extends FrontBaseController
{
    const VIEW = "front.contact.";

    public function index()
    {
        $page = Cache::remember("contact-page", 1440, function () {
            return Page::where("key", "contact-page")->first();
        });

        return view(self::VIEW.__FUNCTION__)
            ->with("page", $page);
    }

    public function contactSubmit(ContactRequest $request)
    {
        $input       = $request->all();
        $input["ip"] = $request->ip();
        $table       = Contact::create($input);

        return redirect()->back()->with("success", "Mesajınız Gönderildi");
    }

    public function newsletterPost(NewsletterRequest $request)
    {
        $input       = $request->all();
        $input["ip"] = request()->ip();
        $table       = Newsletter::create($input);

        return redirect()->back()->with("success", "Abonelik Başvurunuz Alındı");
    }

    public function search(Request $request)
    {
        $searchTerm      = $request->input('key');
        $combinedResults = $this->searchTerm($searchTerm);

        $page = Cache::remember("search-result-page", 1440, function () {
            return Page::where("key", "search-result")->first();
        });

        // Koleksiyonu doğrudan view'a gönder
        return view('front.search')
            ->with("page", $page)
            ->with("combinedResults", $combinedResults);
    }

    public function searchTerm($searchTerm)
    {
        // Gelen key değeri ile cache'e alıyorum ve eğer aynı kelimeyi tekrar aratırsa hızlıca önüne düşüyor saygılar
        return Cache::remember("search-term-".$searchTerm, 1440, function () use ($searchTerm) {
            $combinedResults = new Collection;

            $products = Product::published()->where(function ($query) use ($searchTerm) {
                $query->where('name_'._clang(), 'like', '%'.$searchTerm.'%')
                    ->orWhere('content_'._clang(), 'like', '%'.$searchTerm.'%');
            })->get();

            $categories = Category::where(function ($query) use ($searchTerm) {
                $query->where('name_'._clang(), 'like', '%'.$searchTerm.'%')
                    ->orWhere('content_'._clang(), 'like', '%'.$searchTerm.'%')
                    ->orWhere('short_content_'._clang(), 'like', '%'.$searchTerm.'%');
            })->get();

            $blogs = Article::published()
                ->where(function ($query) use ($searchTerm) {
                    $query->where('name_'._clang(), 'like', '%'.$searchTerm.'%')
                        ->orWhere('content_'._clang(), 'like', '%'.$searchTerm.'%');
                })
                ->whereHas('category', function ($query) {
                    $query->where('parent_id', 3);
                })
                ->get();

            $news = Article::published()->where(function ($query) use ($searchTerm) {
                $query->where('name_'._clang(), 'like', '%'.$searchTerm.'%')
                    ->orWhere('content_'._clang(), 'like', '%'.$searchTerm.'%');
            })->where("article_cat_id", 4)->get();

            $activities = Article::published()->where(function ($query) use ($searchTerm) {
                $query->where('name_'._clang(), 'like', '%'.$searchTerm.'%')
                    ->orWhere('content_'._clang(), 'like', '%'.$searchTerm.'%');
            })->where("article_cat_id", 5)->get();

            $redirects = Redirect::published()->where(function ($query) use ($searchTerm) {
                $query->where("name_"._clang(), "like", "%".$searchTerm."%")
                    ->orWhere("content_"._clang(), "like", "%".$searchTerm."%");
            })->get();

            $faqs = Faq::published()->where(function ($query) use ($searchTerm) {
                $query->where("name_"._clang(), "like", "%".$searchTerm."%")
                    ->orWhere("content_"._clang(), "like", "%".$searchTerm."%");
            })->get();

            // Sonuçları birleştir
            $combinedResults = $combinedResults
                ->merge($faqs)
                ->merge($activities)
                ->merge($blogs)
                ->merge($news)
                ->merge($redirects)
                ->merge($products)
                ->merge($categories);

            return $combinedResults;
        });
    }


}
