<?php

namespace Nette\Upload\Libraries;

use Nette\Upload\Models\Media;
use Config;
use Session;
use \Ramsey\Uuid\Uuid;
use \Symfony\Component\HttpFoundation\File\File as SFile;
use File;
use App\Libraries\Managers\ImageManager;
use Illuminate\Http\UploadedFile;

/**
 * #### Global Upload System Class ####
 *
 * This class manages all uploads and converts them to models.
 *
 * By using session and temp folder manages upload system. Image convertions
 * document uploads and in future video convertions
 *
 *
 * @package App\Libraries\Managers
 */
class UploadManager
{
    /*
     * UPLOAD MANAGER SİSTEM İÇİN upload ları ve sessionları kontrol eden yapıdır
     */

    /**
     * For per Session File Tree it needs a custom UUID v4 key
     * @var
     */
    private $sessionKey;

    /**
     * Sessionless File Vault
     *
     * @var
     */
    private $liveVault;

    /**
     * Errors
     *
     * @var array
     */
    public $errors = [];

    /**
     * Storage Path
     *
     * @deprecated
     * @var
     */
    private $storagePath;

    public $customDate = null;

    /**
     * UploadManager constructor.
     */
    public function __construct()
    {

    }

    /**
     * Set Session Key
     *
     * @param $sessionKey
     */
    public function set($sessionKey)
    {
        $this->sessionKey = $sessionKey ?? '';
    }

    public function setDate($date)
    {
        $this->customDate = $date;
    }

    /**
     * Returns absoulte path of uploaded file.
     *
     * @param $type
     * @return string
     */
    private function getStorageFullPath($type)
    {
        return storage_path($this->getStorageRelativePath($type));
    }

    /**
     * Returns relative type path of uploaded file.
     *
     * @param $type
     * @return string
     */
    private function getStorageRelativePath($type)
    {
        return 'temp/' . $type;
    }

    /**
     * Returns given variable is Uploaded File
     *
     * @param $class
     * @return bool
     */
    private function isUploadedFile($class)
    {
        return is_object($class) && get_class($class) === UploadedFile::class;
    }

    /**
     * Returns addressşing keys
     *
     * @param $type
     * @param $key
     * @return string
     */
    private function sKey($type, $key)
    {
        $sessionKey = $this->sessionKey;
        return "uploadmanager.{$sessionKey}.{$type}.{$key}";
    }

    /**
     * Extracts given files extension.
     *
     * @param $filename
     * @return mixed
     */
    private function extractFileExtension($filename)
    {
        return pathinfo($filename, PATHINFO_EXTENSION);
    }

    /**
     * Moves uploaded file of selected File Vault to designated FS folders
     *
     * When $id provided it will overwrite existing files
     *
     * @param string $type
     * @param array $data
     * @param string $key
     * @param string $method
     * @param null $id
     *
     * @return array|null
     */
    private function moveFile($type, $data, $key, $method = "url", $id = null)
    {

        /*
         *      Diğer fonksiyonların dahilinde çağırılıp içerisine verilen tekil datayı storage lokasyonuna move eder.
         *      Yeni Dosya oluşturur.
         *
         *      ID zorlanmışsa üzerine yazar.
         *
         *      method parametresi file olmayan inputların hangi yordamda işleneceğini belirtir.
         *
         */


        // Burada Önemli olan mevzu datanın payload tek index i var (Gelecek kullanım amaçlı böyle yapı uygun görüldü)
        // "payload" içerisinde dosyayı ya da işlenecek text i barındırıyor.

        $return = [];
        if ($type) {

            //genel kullanım için uuid generate-et
            $uuid = Uuid::uuid4()->toString();

            //ID verildiyse update yapılacaktır. Bu yüzden Generate edilmiş uuid yerine Parametre olarak gelen uuid kullanılır.
            if (!isset($id)) {
                $return["uuid"] = $uuid;
            } else {
                $return["uuid"] = $id;
            }


            // Gelen Datanın Payload ının  ne olduğu önem arzediyor. Çoğu durumda payload http Request ile gelip laravel
            // tarafından dönüştürülmüş UploadedFile class ı oluyor. Ancak bazı durumlarda bu payload datası
            //      + Internet üzerindeki bir absolute url (internet üzerinden dosya çekebilmek için) (Seed edilen Datada çok işe yarıyor)
            //      + Filesystem Üzerinden bir Absolute path (Sistem üzerindeki klasörlerden birinden yararlanabilmek için) (Diğer sistemlerden aktarım sırasında işe yarıyor)
            //      + base64 ile encode edilmiş bir imaj datası  (Cropper ve Diğer programlarla direkt etkileşim için)
            // olabiliyor. Bu durumu kontrol ederek hangisinin çalıştırılacağına dair bir kontrol yapmalıyız.

            //payload text mi UploadedFile mı ?
            if ($this->isUploadedfile($data["payload"])) {

                #region UploadedFile

                // fileName extract et yoksa hata ver
                $return["clientName"] = $data["payload"]->getClientOriginalName();
                if (!($return['clientName'])) {
                    $return["failed"] = true;
                    $return["error"] = "fileName doesnt exists";
                    return $return;
                }


                //fileExt extract et yoksa hata ver
                $fileExt = $data["payload"]->getClientOriginalExtension();
                if (!($fileExt)) {
                    $return["failed"] = true;
                    $return["error"] = "fileExt doesnt exists";
                    return $return;
                }

                //file extension kabul edilebilen uzantılar arasında mı
                if (!in_array($fileExt, config("upload.{$type}.allowedExts"))) {
                    $return["failed"] = true;
                    $return["error"] = "fileExt not allowed";
                    return $return;
                }

                //$type img ise gelen resmin exit türünün config/upload.php içerisinde belirlenen exif tipleri içinde olması gerek
                if ($type == 'img') {
                    if (!in_array(exif_imagetype($data['payload']->path()), config("upload.{$type}.allowedExifTypes"))) {
                        $return["failed"] = true;
                        $return["error"] = "filetype not valid";
                        return $return;
                    }
                }


                //mimeType ı çek (Opsiyonel. Olup olmamasının çok bi anlamı yok)
                //$mimeType = $data["info"]["mime"];
                //$return["mime"] = $mimeType;

                // TODO: extension check // FileSize Check // Bomba Check // Type Check

                // Herhangi bir patlama fatal olmamalı
                try {

                    // Upload edilen dosyayı temp klasöründen alıpğ uuid.extension ismiyle storage klasörüne
                    // relative şekilde at
                    $tempFile = $data["payload"]->move(
                        $this->getStorageFullPath($type),
                        $return["uuid"] . (isset($fileExt) && ($fileExt != "") ? '.' . $fileExt : '')
                    );

                    //UUID yi key olarak kullanarak return Datasını Güncelle
                    $return["filename"] = $tempFile->getFilename(); //Upload Edilen İsmi
                    $return["ext"] = $fileExt; //Extract Edilmiş Dosya Uzantısı
                    $return["key"] = $key;     //Dosya Tipi
                    $return["storageRelativePath"] = $this->getStorageRelativePath($type); //Storage İçindeki Relative Path i
                    $return["storageFullPath"] = $tempFile->getRealPath(); //
                    $return["size"] = $tempFile->getSize(); //File Size

                } catch (\Exception $ex) {
                    // Exception ı error a ekleyip Dosya Bazında Hata Kontrolü ile bitir
                    $return["failed"] = true;
                    $return["error"] = "Error Occured When image Uploading : " . $ex->getMessage();

                    return $return;
                }

                #endregion

            } else if (is_string($data["payload"])) {

                //region Other Data Methods

                // Buraya gelen datanın string olduğu anlaşılıyor. moveFile fonksiyonunun 3. paramtresi
                // uploadedFile olmayan data için hangi metodu kullanacağımızı belirtiyor
                // Varsayılan metod url

                switch ($method) {
                    case "url":
                        //Dosyayı Internet veya Intranet üzerindeki bir sunucudan çek.
                        try {
                            $tempFileName = $this->getStorageFullPath($type) . DIRECTORY_SEPARATOR . $return["uuid"];
                            $return["ext"] = $this->getFileExtension($data["payload"]);
                            file_put_contents($tempFileName, file_get_contents($data["payload"]), false, stream_context_create(['ssl' => [
                                "verify_peer" => false,
                                "verify_peer_name" => false,
                            ]]));
                            //TODO: extension check // FileSize Check // Type Check
                            $tempFile = new SFile($tempFileName);
                        } catch (Exception $ex) {
                            Log::error("Error Occured When image Uploading from URL : " . $ex->getMessage());
                            $return = false;
                        }

                        break;
                    case "fs":
                        //Dosyayı filesystem üzerinden verilen absolute path kullanarak çek
                        try {
                            //TODO: extract filename & ext vs
                            $tempFileName = $this->getStorageFullPath($type) . DIRECTORY_SEPARATOR . $return["uuid"];
                            //TODO: extension check // FileSize Check // Type Check
                            File::copy($data['payload'], $tempFileName);
                            $tempFile = new SFile($tempFileName);
                            if ($data['payload'] ?? null)
                                if (File::exists($data['payload'])) {
                                    $tempFileName = $this->getStorageFullPath($type) . DIRECTORY_SEPARATOR . $return["uuid"];
                                    $isExtensionExists = pathinfo($tempFileName, PATHINFO_EXTENSION);
                                    if ($isExtensionExists == '') {
                                        $supposedExtension = pathinfo($data['payload'], PATHINFO_EXTENSION);
                                        if ($supposedExtension) {
                                            $tempFileName = $this->getStorageFullPath($type) . DIRECTORY_SEPARATOR . $return["uuid"] . '.' . $supposedExtension;
                                        } else {
                                            $detectedMimeType = mime_content_type($data['payload']);
                                            $fileExtension = getFileExtensionWithType($detectedMimeType);
                                            $tempFileName = $this->getStorageFullPath($type) . DIRECTORY_SEPARATOR . $return["uuid"] . '.' . $fileExtension;
                                        }
                                    }
                                    //TODO: extension check // FileSize Check // Type Check
                                    File::copy($data['payload'], $tempFileName);
                                    $tempFile = new SFile($tempFileName);
                                } else {
                                    throw new \Exception('File Not exits');
                                }
                            else {
                                throw new \Exception('payload empty');
                            }

                        } catch (Exception $ex) {
                            Log::error("Error Occured When image Uploading from FileSystem : " . $ex->getMessage());
                            $return = false;
                        }

                        break;
                    case "base64":
                        // DOsyayı data olarak gönderilen base64 stringinden generate et
                        try {
                            $tempFileName = $this->getStorageFullPath($type) . DIRECTORY_SEPARATOR . $return["uuid"];
                            $isExtensionExists = pathinfo($tempFileName, PATHINFO_EXTENSION);

                            if ($isExtensionExists == '') {
                                preg_match("/^data:(.*);base64/s", $data['payload'], $fileExtension);
                                if (isset($fileExtension[1])) {
                                    $fileExtension = getFileExtensionWithType($fileExtension[1]);
                                }
                                $tempFileName = $this->getStorageFullPath($type) . DIRECTORY_SEPARATOR . $return["uuid"] . '.' . $fileExtension;
                            }


                            file_put_contents($tempFileName, base64_decode(preg_replace('#data:image/[^;]+;base64,#', '', str_replace(' ', '+', $data["payload"]))));
                            //TODO: extract ext vs
                            $tempFile = new SFile($tempFileName);
                        } catch (Exception $ex) {
                            Log::error("Error Occured When image Uploading from Base64 : " . $ex->getMessage());
                            $return = false;
                        }
                        break;

                }

                $return['clientName'] = ($method != 'base64') ? $data["payload"] : 'base64';
                $return["filename"] = $tempFile->getFilename();
                if (!isset($return["ext"]) || $return['ext'] == '') {
                    $return["ext"] = $this->getFileExtension($tempFile->getFilename()); //Ext from Up
                }
                $return["key"] = $key;
                $return["storageRelativePath"] = $this->getStorageRelativePath($type);
                $return["storageFullPath"] = $tempFile->getRealPath();
                $return["size"] = $tempFile->getSize();
                //endregion

            }
        } else {
            $return["failed"] = true;
            $return["error"] = "Type Not Correct";
        }

        return $return;


    }

    public function getFileExtension($url)
    {
        $ext = 'jpg';
        $extension = pathinfo($url, PATHINFO_EXTENSION);
        if ($extension != '') {
            $ext = $extension;
        }

        return $ext;
    }

    /**
     * Adds a file to Session File Tree by using move() method
     *
     * @param string $type
     * @param string $key
     * @param array $input
     * @param null $templateID
     * @param bool $multiple
     * @param string $method
     * @return array|\Illuminate\Session\SessionManager|\Illuminate\Session\Store|mixed
     */
    public function add($type, $key, $input, $templateID = null, $multiple = false, $method = 'url')
    {
        /*
         *      add fonksiyonu kendisine belirtilen tip, key, input ve method için gerekli session datasını manipule eder
         *      Gelen Inputun dizi olması halinde her elemanı moveFile metodundan geçirir.
         *      Olmadığı takdirde tek input u moveFile a gönderir
         *
         *      Dönen sonuca göre diziyi editleyerek format bilgisi ile birlikte tekrar session a atar.
         *
         *      Ayrıca Max file kontrollerini gerçekleştirir. ve Single File kontrollerini gerçekleştirir.
         *
         *      Geriye Dönen hatalar için Sınıfın errors nesnesini doldurur.
         *
         */


        if (!$this->sessionKey) {
            throw new \Error('session Key Not Specified');
        }

        //Format config içerisinden çekilecek media format datası config("upload.{$type}.keys.{$key}");
        $format = config("upload.{$type}.keys.{$key}");

        //Varolan Session Datasını çek.
        // Bu data aşağıdaki Kontrol yapısında manipüle edilerek tekrar session a atılacak
        $sessionData = session($this->sKey($type, $key));
        if (is_null($sessionData))
            $sessionData = [];

        //Payload Çoklu mu Tekli mi ? Bunu gönderen Controller Belirliyor
        if ($multiple) {

            // TODO: multiple file input revize edilecek. TemplateID ile uydurulacak
            // INPUT bir multi array

            // Dizinin yapısı

            /*
             *  $input = [
             *              ['payload' => ORJINAL DATA],
             *              ['payload' => ORJINAL DATA],
             *              ['payload' => ORJINAL DATA],
             *           ];
             */

            //Bu durumda Her bir eleman moveFile fonksiyonuna tekmiş gibi verilmeli.

            $count = count(array_except($sessionData, ['failed', 'error']));
            //Tüm elemanlarını işleme koy
            foreach ($input as $data) {

                //TODO: respect file limits from config
                if ($format["max"] > 0 && $count == $format["max"]) {
                    $this->errors[$type][$key][] = 'Max File Reached';
                    break;
                }

                //Elemanı İşleme Koy
                $uploadedFileData = $this->moveFile($type, $data ?? [], $key, $method);
                $generatedTemplateID = Uuid::uuid4()->toString();
                if (isset($uploadedFileData["failed"]) && $uploadedFileData["failed"]) {
                    $this->errors[$type][$key][$uploadedFileData["clientName"]] = $uploadedFileData["error"];
                } else {
                    $sessionData[$uploadedFileData["uuid"]] = [
                        'templateID' => $generatedTemplateID,
                        'format' => $format,
                        'file' => $uploadedFileData
                    ];
                }
                $count++;
            }
        } else {
            // Single Geldi

            //Input payload içeriyor mu?
            if ($input["payload"]) {
                //Format max 1 ise session u replace etmeliyiz. Bu sayade Hata almadan ve her seferinde delete etmeden session u update edebiliriz.
//                if ($format["max"] == 1) {
//                    $sessionData = [];
//                }

                $count = count(array_except($sessionData, ['failed', 'error']));

                if (isset($count)) {
                    //max 0 dan büyük. Count dan Küçük. Dosya eklenebilir.
                    //TODO : Respect File Limits from config if exceeds replace file

                    //Elemanı İşleme Koy

                    $data = $input;

                    $uploadedFileData = $this->moveFile($type, $data, $key, $method);
                    if (isset($uploadedFileData["failed"]) && $uploadedFileData["failed"]) {
                        $this->errors[$type][$key][$uploadedFileData["clientName"]] = $uploadedFileData["error"];
                    } else {
                        $sessionData[$uploadedFileData["uuid"]] = ['templateID' => $templateID, 'format' => $format, 'file' => $uploadedFileData];
                    }


                } else if (!isset($format["max"]) || (isset($format["max"]) && $format["max"] == 0)) {
                    //Max 0 olarak belirlenmiş ya da tanımlanmamış. Dosya Eklenebilir.
                    $data = $input;

                    $uploadedFileData = $this->moveFile($type, $data, $key, $method);
                    if (isset($uploadedFileData["failed"]) && $uploadedFileData["failed"]) {
                        $this->errors[$type][$key][$uploadedFileData["clientName"]] = $uploadedFileData["error"];

                    } else {

                        $sessionData[$uploadedFileData["uuid"]] = ['templateID' => $templateID, 'format' => $format, 'file' => $uploadedFileData];
                    }
                } else {
                    $this->errors[$type][$key][] = 'Max File Reached';

                }
            }
        }

        $this->liveVault = $sessionData;
        session([$this->sKey($type, $key) => $sessionData]);
        return $this->liveVault;

    }

    /**
     * File Processing class Map with format $type => $class
     *
     * @var array
     */
    public $keyManagers = [
        'img' => 'Nette\Upload\Libraries\ImageManager',
        'doc' => 'Nette\Upload\Libraries\DocumentManager',
        'vid' => 'Nette\Upload\Libraries\VideoManager',
    ];

    /**
     * Saves All files in Session File Tree for given Session keys
     * Returns All Media Models in array
     *
     * @return array
     */
    public function saveAll()
    {

        if (!$this->sessionKey) {
            throw new \Error('session Key Not Specified');
        }

        $collection = [];

        $sessionKey = $this->sessionKey;
        $types = session("uploadmanager.{$sessionKey}");

        foreach ($types as $type_key => $keys) {

            foreach ($keys as $key_key => $itemlist) {
                if ($itemlist) {
                    foreach ($itemlist as $item_key => $item) {
                        if (in_array($type_key, array_keys($this->keyManagers))) {
                            $manager = new $this->keyManagers[$type_key]($item);
                            $savedItem = $manager->save();
                            if ($savedItem) {
                                $collection[$type_key][$key_key][] = $savedItem;
                            }
                        }
                    }
                }

            }


        }

        return $collection;


    }

    /**
     * Save a single branch of File Tree by given file type
     *
     * Returns models in array
     *
     * @param      $type
     * @param      $key
     * @param bool $live
     * @return array|bool|null
     */
    public function save($type, $key, $live = false)
    {

        $sessionKey = $this->sessionKey;
        if (!$sessionKey) {
            throw new \Error('session Key Not Specified');
        }


        $tkey = "uploadmanager.{$sessionKey}.{$type}.{$key}";
        $items = session($tkey);

        // Bazı durumlarda session ı kullanamıyoruz. Bu session ı kullanamadığımız durumlarda
        // livaVault adındaki bir değişken daha class aktifken dosyayı tutuyor.
        // ve live parametresiyle bu değişkeni işleyip işlemediğimizi hesaplıyoruz

        if ($live) {
            $items = $this->liveVault ?? null;
        }

        // Upload Manager Vault u initialize ettiyse ve boş bir array geliyorsa
        // Kullanıcı en azından Delete fonksiyonunu çalıştırmış demektir.
        // Bu durumda kullanıcıya false dönmek yerine null dönmemiz gerekiyor.

        if (is_array($items) && count($items) < 1) {
            return null;
        }

        // Vault boş bir array değil.
        // Vault null ise bu vault un hiç initialize edilmediği anlamına gelir.
        // Bu durumda false dönmeli.
        // Vault doluysa içinde itemler var demektir.
        // Aksi takdirde configdeki max değerine bakılıp 1 ise direk nesneyi değilse nesnelerin hepsi içinde olacak şekilde diziyi döndürmeli
        if ($items) {
            if (config("upload.{$type}.keys.{$key}.max") == 1) {
                if (in_array($type, array_keys($this->keyManagers))) {
                    $manager = new $this->keyManagers[$type](array_first($items), $this->customDate);
                    $savedItem = $manager->save();
                    if ($savedItem) {
                        //Empty Sessions
                        session()->forget("uploadmanager.{$sessionKey}.{$type}.{$key}");
                        return $savedItem;
                    }
                } else return [];

            } else {

                $collection = [];
                foreach ($items as $item_key => $item) {
                    if (in_array($type, array_keys($this->keyManagers))) {
                        $manager = new $this->keyManagers[$type]($item, $this->customDate);
                        $savedItem = $manager->save();

                        if ($savedItem) {
                            unset($items[$item_key]);
                            $collection[] = $savedItem;
                        }
                    }
                }

                //Update Sessions
                session(["uploadmanager.{$sessionKey}.{$type}.{$key}" => $items]);
                return $collection;

            }
        } else return false;

    }

    /**
     * Get Single File data from File tree.
     *
     * if its type is img or doc with png extension this method also returns preview of uploaded file
     *
     *
     * @param      $type
     * @param null $key
     * @param null $id
     * @return bool|\Illuminate\Session\SessionManager|\Illuminate\Session\Store|mixed
     */
    public function get($type, $key = null, $id = null)
    {

        $sessionKey = $this->sessionKey;

        if (!$sessionKey) {
            throw new \Error('session Key Not Specified');
        }

        $ret = session("uploadmanager.{$sessionKey}.{$type}");

        if (isset($key)) {
            $ret = session("uploadmanager.{$sessionKey}.{$type}.{$key}");


            if (isset($id)) {
                $ret = session("uploadmanager.{$sessionKey}.{$type}.{$key}.{$id}");

                if ($ret == null) return false;
                if ($type == 'img') {
                    $manager = new $this->keyManagers[$type]($ret);
                    $ret["preview"] = $manager->getPreview();
                }
            }
        }

        return $ret;
    }

    /**
     * Deletes an end node from Filetree
     *
     *
     * @param $type
     * @param $key
     * @param $id
     * @return bool
     */
    public function delete($type, $key, $id)
    {
        //Varolan Session Datasını çek.
        // Bu data aşağıdaki Kontrol yapısında varyanse edilerek tekrar session a atılacak

        $sessionData = session("uploadmanager.{$this->sessionKey}.{$type}.{$key}", []);

        // Session Yoksa delete isteği gönderildiğinde boş array atanmalı.
        // Bu sayede array mi null mu kontrolü yapılarak varolan resim nullable edilebilir
//        if($sessionData == null){
//            session([$this->sKey($type, $key) => []]);
//        }
        unset($sessionData[$id]);
        session([$this->sKey($type, $key) => $sessionData]);

        $ret = session("uploadmanager.{$this->sessionKey}.{$type}.{$key}.{$id}") == null;
        return result($ret);

    }

    public function deleteAll($type, $key)
    {
        session([$this->sKey($type, $key) => null]);
        $ret = session("uploadmanager.{$this->sessionKey}.{$type}.{$key}") == null;
        return result($ret);
    }

    /**
     * Update single file from File Tree
     *
     *
     * @param        $type
     * @param        $key
     * @param        $id
     * @param null $data
     * @param string $method
     * @return bool
     */
    public function update($type, $key, $id, $data = null, $method = 'url')
    {
        $ret = true;
        $format = config("upload.{$type}.keys.{$key}");

        $sessionData = session($this->sKey($type, $key) . '.' . $id);

        if (is_null($sessionData))
            $sessionData = null;

        if ($data) {
            $uploadedFileData = $this->moveFile($type, $data, $key, $method, $id);
            if (isset($uploadedFileData["failed"]) && $uploadedFileData["failed"]) {
                $this->errors[$type][$key][$uploadedFileData["clientName"]] = $uploadedFileData["error"];
            } else {
                $sessionData = ['format' => $format, 'file' => $uploadedFileData];
            }
        }

        session([$this->sKey($type, $key) . '.' . $id => $sessionData]);
        return $ret;

    }

    /**
     * Get File Tree İtem Count of given key
     *
     * @param $type
     * @param $key
     * @return int
     */
    public function getCount($type, $key)
    {

        if (!$this->sessionKey) {
            throw new \Error('session Key Not Specified');
        }

        $ret = session("uploadmanager.{$this->sessionKey}.{$type}.{$key}", []);
        return count($ret);
    }

    /**
     * Returns if has any items
     *
     * @param $type
     * @param $key
     * @return bool
     */
    public function hasItem($type, $key)
    {

        if (!$this->sessionKey) {
            throw new \Error('session Key Not Specified');
        }

        $ret = session("uploadmanager.{$this->sessionKey}.{$type}.{$key}", []);
        return count($ret ?? []) > 0;
    }

    //region Will Implement

    /**
     * @param $data
     * @todo
     */
    public function addImage($data)
    {
    }

    /**
     * @param $key
     * @todo
     */
    public function deleteImage($key)
    {
    }

    /**
     * @param $key
     * @param $data
     * @todo
     */
    public function updateImage($key, $data)
    {
    }

    /**
     * @param $data
     * @todo
     */
    public function addDocument($data)
    {
    }

    /**
     * @param $key
     * @todo
     */
    public function deleteDocument($key)
    {
    }

    /**
     * @param $key
     * @param $data
     * @todo
     */
    public function updateDocument($key, $data)
    {
    }

    /**
     * @param $data
     * @todo
     */
    public function addVideo($data)
    {
    }

    /**
     * @param $key
     * @todo
     */
    public function deleteVideo($key)
    {
    }

    /**
     * @param $key
     * @param $data
     * @todo
     */
    public function updateVideo($key, $data)
    {
    }
    //endregion
}
