<?php

namespace Nette\Upload\Libraries;


use Config;
use Nette\Upload\Models\Media;
use File;
use Image;
//use ImageOptimizer;
use Ramsey\Uuid\Uuid;
use Log;
use Storage;

/**
 * #### Process Images and Create Associated Models from them ####
 * @package App\Libraries\Managers
 */
class ImageManager
{
    public $image;
    public $format;
    public $model;
    public $type = 'img';

    public $disk = 'upload';
    public $diskOriginal = 'upload-original';

    private $uploadPath = null;
    private $ext = null;

    private $customDate = null;

    public function __construct($image,$customDate=null)
    {

        $this->image = $image['file'];
        $this->format = $image['format'];
        $this->uploadPath = config("upload.img.uploadPath");
        $this->ext = null;
        $this->disk = config('upload.disk');
        $this->diskOriginal = config('upload.originalDisk');

        $this->customDate = $customDate;
    }

    private function determineFolder()
    {
        if($this->customDate == null) {
            return date("Y") . DIRECTORY_SEPARATOR . date('m', time()) . DIRECTORY_SEPARATOR . date('d', time());
        } else {
            return date("Y",$this->customDate) . DIRECTORY_SEPARATOR . date('m', $this->customDate) . DIRECTORY_SEPARATOR . date('d', $this->customDate);
        }
    }

    private function determineOriginalFolder()
    {
        if($this->customDate == null) {
            return date("Y") . DIRECTORY_SEPARATOR . date('m', time()) . DIRECTORY_SEPARATOR . date('d', time());
        } else {
            return date("Y",$this->customDate) . DIRECTORY_SEPARATOR . date('m', $this->customDate) . DIRECTORY_SEPARATOR . date('d', $this->customDate);
        }
    }

    private function error($message)
    {
        Log::error('ImageManager: ' . $message);
    }

    public function save()
    {

        //Config Check
        if (!$this->uploadPath) {
            $this->error("ImageUpload Path Undefined");
            return false;
        }

        //Photo Config Check
        if (!$this->checkConfig()) {
            $this->error("Image size config error");
            return false;
        }

        $image = $this->image;

        $modelUuid = Uuid::uuid4()->toString();

        //Determine Target Folder
        $targetSubFolder = $this->determineFolder();
        //Specify folder field data
        $uploadDir = $this->uploadPath . $targetSubFolder;
        $originalDir = $this->uploadPath . $targetSubFolder;


        $model = Media::create([
            'uuid' => $modelUuid,
            'folder' => $targetSubFolder,
            'ext' => $image["ext"],
            'type' => $this->type,
            'key' => $image["key"],
            'name' => $image['filename'],
            'size' => 0
        ]);

        if ($model) {
            $savedImage = $this->saveImage($uploadDir, $originalDir, $image, $model);
            if ($savedImage) {
                return $model;
            } else {
                return false;
            }
        } else {
            return false;
        }

    }

    private function checkConfig()
    {
        $check = true;

        $data = $this->format;

        if ($data) {
            foreach ($data as $k => $v) {
                if (isset($v["variations"]) && count($v["variations"]) > 0) {
                    foreach ($v["variations"] as $key => $val) {

                        if ($val["width"] > $v["width"]) {
                            Log::error("Image Variation Size Error on width : " . $k . ' - ' . $key);
                            $check = false;
                        }

                        if ($val["height"] > $v["height"]) {
                            Log::error("Image Variation Size Error on height : " . $k . ' - ' . $key);
                            $check = false;
                        }

                        if ($key == "original") {
                            Log::error("Image Variation Name Cannot be original: " . $k . ' - ' . $key);
                            $check = false;
                        }
                    }
                }
            }
            //Height or Width Ayarlanmadıysa bunları null olarak atamalı
            if (!isset($data["width"])) {
                $data["width"] = null;
            }

            if (!isset($data["height"])) {
                $data["height"] = null;
            }


            $this->format = $data;


        }

        return $check;

    }

    private function checkFolder($path, $disk)
    {

        if (!Storage::disk($disk)->exists($path)) {
            try {
                Storage::disk($disk)->makeDirectory($path);
            } catch (\Exception $ex) {
                Log::warning("Couldn't create folder $path on disk $disk");
                return false;
            }
        }


        if (!Storage::disk($this->disk)->exists($path)) {
            try {
                Storage::disk($this->disk)->makeDirectory($path);
            } catch (\Exception $ex) {
                Log::warning("Couldn't create folder $path.");
                return false;
            }
        }

        return true;
    }

    private function saveImage($uploadDir, $originalDir, $tempFile, $model)
    {
        try {
            if ($this->checkFolder($uploadDir, $this->disk) && $this->checkFolder($originalDir, $this->diskOriginal)) {
                $return = $this->processImage($tempFile, $uploadDir, $originalDir, $model, $this->format);

                $willEncode = (bool)($this->format['encode'] ?? true);
                if ($willEncode) {
                    $extension = $this->ext($this->extension($this->format), true);
                } else {
                    $extension = $tempFile["ext"];
                }

                $model->ext = $extension;
                $model->save();


                @unlink($tempFile['storageFullPath']);
            } else {
                $return = false;
            }

            return $return;
        } catch (Exception $ex) {
            Log::error("Error Occured while image saving to FileSystem : " . $ex->getMessage());
            $return = false;
        };

    }

    private function fullModelPath($targetSubDir, $modelid)
    {
        return $targetSubDir . DIRECTORY_SEPARATOR . $modelid;
    }

    private function processImage($tempFile, $targetDir, $originalDir, $model, $format, $preview = false)
    {
        $config = $format;

        //Set Local Control Mechanisms
        $willEncode = false;
        $willResize = false;
        $resizeStrategy = null;
        $resizeMethod = null;


        $hasVariations = false;

        //Will we encode this image
        $willEncode = (bool)($config['encode'] ?? true);

        //Will we resize this image
        $willResize = ((bool)($config['max-height'] ?? true)) ||
            ((bool)($config['max-width'] ?? true)) ||
            ((bool)($config['height'] ?? true)) ||
            ((bool)($config['width'] ?? true));


        //Set Resize Method
        // 1 Absolute , 2 Optional, 3-noresize
        if ($willResize) {
            $resizeStrategy = ((!is_null($config['max-height'] ?? null)) || (!is_null($config['max-width'] ?? null))) ? 2 : $resizeStrategy;
            $resizeStrategy = ((!is_null($config['height'] ?? null)) || (!is_null($config['width'] ?? null))) ? 1 : $resizeStrategy;

        }

        //Will this image has variations
        $hasVariations = isset($config["variations"]) && count($config["variations"]) > 0;


        $tempImage = Image::make($tempFile['storageFullPath']);
//        if(array_key_exists('watermark', $format) && array_key_exists('source', $format['watermark'])) {
//            $tempImage = $tempImage->insert($format['watermark']['source'], array_key_exists('position', $format['watermark']) ? $format['watermark']['position'] : 'center', array_key_exists('offset_x', $format['watermark']) ? $format['watermark']['offset_x'] : 0,array_key_exists('offset_y', $format['watermark']) ? $format['watermark']['offset_y'] : 0);
//        }
        //TODO: image Extension Detection
        $tempImageExtension = $this->ext(($tempFile['ext'] ?? $this->iext(true)), true);

        $isItGif = (strtolower($tempImageExtension) == 'gif');

        if ($willEncode && $isItGif) {
            $isItGif = false;
            $tempImageExtension = $this->extension(null);
        }

        //region Temp File Information
        if (!$isItGif) {
            $tempImageW = $tempImage->width();
            $tempImageH = $tempImage->height();
            $tempImageFilesize = $tempImage->filesize();
            $tempImageRatio = $tempImageW / $tempImageH;
        } else {
            list($gWidth, $gHeight) = getimagesize($tempFile['storageFullPath']);
            $gSize = filesize($tempFile['storageFullPath']);
            $tempImageW = $gWidth;
            $tempImageH = $gWidth;
            $tempImageFilesize = $gSize;
            $tempImageRatio = $tempImageW / $tempImageH;
        }
        //endregion

        //region Save Original File


        //IF preview mode is true dont save anything
        if (!$preview) {

            //EncodedOriginalFileName
            $eofn = $this->fullModelPath($originalDir, $model->uuid) . '-original' . $this->ext($this->extension($config));
            //NonEncodedOriginalFileName
            $neofn = $this->fullModelPath($originalDir, $model->uuid) . '-original' . $this->ext($tempImageExtension);

            if ($willEncode) {
                //This will encoded to different format
                Storage::disk($this->diskOriginal)->put($eofn, (string)$tempImage->encode($this->extension($config), 100));
            } else {
                //No Encoding
                if ($isItGif) {
                    //IF its a gif we cannot save it with imagemanager. We will save it as it is
                    Storage::disk($this->diskOriginal)->put($neofn, \File::get($tempFile['storageFullPath']));
                } else {
                    //Just Empty encode and save it
                    Storage::disk($this->diskOriginal)->put($neofn, (string)$tempImage->encode());
                }

            }
        } // Preview Mod
        else {
            if ($willEncode) {
                //Just Empty encode
                $tempImage->encode($this->extension($config), 100);
            }
        }
        //endregion

        //region Resize Image
        if ($willResize) {
            //We Will Resize this image
            switch ($resizeStrategy) {
                case 1:
                    switch ($config['method'] ?? 'auto') {
                        case 'crop':
                            $cropWidth = (int)((isset($config['width'])) ? ($config['width']) : ((isset($config['height'])) ? ($tempImageW / ($tempImageH / $config['height'])) : (1)));
                            $cropHeight = (int)((isset($config['height'])) ? ($config['height']) : ((isset($config['width'])) ? ($tempImageH / ($tempImageW / $config['width'])) : (1)));

                            $tempImage->crop($cropWidth, $cropHeight);
                            break;
                        case 'resize':
                            $resizeWidth = (isset($config['width']) ? ((int)($config['width'])) : null);
                            $resizeHeight = (isset($config['height']) ? ((int)($config['height'])) : null);

                            $tempImage->resize($resizeWidth, $resizeHeight, function ($c) use ($resizeWidth, $resizeHeight) {
                                if (is_null($resizeWidth) || is_null($resizeHeight)) {
                                    $c->aspectRatio();
                                }
                            });
                            break;
                        case 'auto':
                            $fitWidth = (int)((isset($config['width'])) ? ($config['width']) : ((isset($config['height'])) ? ($tempImageW / ($tempImageH / $config['height'])) : (1)));
                            $fitHeight = (int)((isset($config['height'])) ? ($config['height']) : ((isset($config['width'])) ? ($tempImageH / ($tempImageW / $config['width'])) : (1)));

                            $tempImage->fit($fitWidth, $fitHeight);
                            break;

                    }

                    break;
                case 2:
                    //Get Max Dimensions of image
                    $maxWidth = (isset($config['max-width']) ? ((int)($config['max-width'])) : null);
                    $maxHeight = (isset($config['max-height']) ? ((int)($config['max-height'])) : null);

                    //If Max width defined and image is wider than that
                    if ($maxWidth) {
                        if ($tempImage->width() > $maxWidth) {
                            $tempImage->resize($maxWidth, null, function ($c) {
                                $c->aspectRatio();
                            });
                        }
                    }

                    //If Max height defined and image is taller than that
                    if ($maxHeight) {
                        if ($tempImage->height() > $maxHeight) {
                            $tempImage->resize(null, $maxHeight, function ($c) {
                                $c->aspectRatio();
                            });
                        }
                    }


                    break;
                default;
                    break;
            }

        }
        //endregion

        //region Serve Preview and End
        if ($preview) {
            if (!$willEncode) {
                if ($isItGif) {
                    $tempImageData = base64_encode(file_get_contents($tempFile['storageFullPath'], false, stream_context_create(['ssl' => [
                        "verify_peer" => false,
                        "verify_peer_name" => false,
                    ]])));
                    $tempImageSrc = 'data: ' . mime_content_type($tempFile['storageFullPath']) . ';base64,' . $tempImageData;
                    return $tempImageSrc;

                }
            }
            return $tempImage->encode('data-url')->encoded;
        }
        //endregion

        //region Save Main File
        $return = null;
        //EncodedFileName
        $efn = $this->fullModelPath($targetDir, $model->uuid) . $this->ext($this->extension($config));
        //NonEncodedFileName
        $nefn = $this->fullModelPath($targetDir, $model->uuid) . $this->ext($tempImageExtension);
        //SAVE FILE
        if ($willEncode) {
            //This will encoded to different format
            $return = Storage::disk($this->disk)->put($efn, (string)$tempImage->encode($this->extension($config), 100));
        } else {
            //No Encoding
            if ($isItGif) {
                //IF its a gif we cannot save it with imagemanager. We will save it as it is
                $return = Storage::disk($this->disk)->put($nefn, \File::get($tempFile['storageFullPath']));
            } else {
                //Just Empty encode and save it
                $return = Storage::disk($this->disk)->put($nefn, (string)$tempImage->encode());
            }

        }
        //endregion

//        ImageOptimizer::optimize(storage_path($this->disk . '/' . $nefn));

        if (!$hasVariations) {
            return $return;
        }

        $tempImage->destroy();
        unset($tempFile);

        //region Variations
        if (!$isItGif) {
            $mainFileOfVariations = $willEncode ? $efn : $nefn;
            foreach ($config["variations"] as $key => $var) {
                $tempVariationFile = Image::make(Storage::disk($this->disk)->get($mainFileOfVariations));

                $evfn = $this->fullModelPath($targetDir, $model->uuid) . '-' . $key . $this->ext($this->extension($config));
                $nevfn = $this->fullModelPath($targetDir, $model->uuid) . '-' . $key . $this->ext($tempImageExtension);
                if ($tempVariationFile) {
                    $variationResizeStrategy = 0;
                    $variationResizeStrategy = ((!is_null($var['max-height'] ?? null)) || (!is_null($var['max-width'] ?? null))) ? 2 : $variationResizeStrategy;
                    $variationResizeStrategy = ((!is_null($var['height'] ?? null)) || (!is_null($var['width'] ?? null))) ? 1 : $variationResizeStrategy;

                    //We Will Resize this image
                    switch ($variationResizeStrategy) {
                        case 1:
                            switch ($var['method'] ?? 'auto') {
                                case 'crop':
                                    $cropWidth = $var['width'] ?? $tempVariationFile->width();
                                    $cropHeight = $var['height'] ?? $tempVariationFile->height();

                                    $tempVariationFile->crop($cropWidth, $cropHeight);
                                    break;
                                case 'resize':
                                    $resizeWidth = (isset($var['width']) ? ((int)($var['width'])) : null);
                                    $resizeHeight = (isset($var['height']) ? ((int)($var['height'])) : null);

                                    $tempVariationFile->resize($resizeWidth, $resizeHeight, function ($c) use ($resizeWidth, $resizeHeight) {
                                        if (is_null($resizeWidth) || is_null($resizeHeight)) {
                                            $c->aspectRatio();
                                        }
                                    });
                                    break;
                                case 'auto':
                                    $fitWidth = $var['width'];
                                    $fitHeight = $var['height'];

                                    $tempVariationFile->fit($fitWidth, $fitHeight);
                                    break;

                            }

                            break;
                        case 2:
                            //Get Max Dimensions of image
                            $maxWidth = (isset($var['max-width']) ? ((int)($var['max-width'])) : null);
                            $maxHeight = (isset($var['max-height']) ? ((int)($var['max-height'])) : null);

                            //If Max width defined and image is wider than that
                            if ($maxWidth) {
                                if ($tempVariationFile->width() > $maxWidth) {
                                    $tempVariationFile->resize($maxWidth, null, function ($c) {
                                        $c->aspectRatio();
                                    });
                                }
                            }

                            //If Max height defined and image is taller than that
                            if ($maxHeight) {
                                if ($tempVariationFile->height() > $maxHeight) {
                                    $tempVariationFile->resize(null, $maxHeight, function ($c) {
                                        $c->aspectRatio();
                                    });
                                }
                            }


                            break;
                        default;
                            break;
                    }
                    if ($willEncode) {
                        Storage::disk($this->disk)->put($evfn, (string)$tempVariationFile->encode($this->extension($config), 100));
                    } else {
                        Storage::disk($this->disk)->put($nevfn, (string)$tempVariationFile->encode());
                    }

                }
                //variation optimizer

//                ImageOptimizer::optimize(storage_path($this->disk . '/' . $nevfn));
            }
        }
        //endregion

        return $return;


    }

    private function iext($withOutDot = false)
    {
        return (($withOutDot) ? '' : '.') . $this->ext;
    }

    private function ext($ext, $withOutDot = false)
    {
        return (($withOutDot) ? '' : '.') . $ext;
    }

    private function extension($config)
    {
        if (isset($config['encode']) && $config['encode']) {

            if ($config['encode-to']) {
                return $config['encode-to'];
            }

            return config('upload.img.defaultExt');
        }

        return config('upload.img.defaultExt');
    }

    private function previewImage($tempFile, $format)
    {
        return $this->processImage($tempFile, null, null, null, $format, true);

    }

    public function getPreview()
    {
        //Config Check
        if (!Config::get("upload.img.uploadPath")) {
            Log::error("Upload Path Undefined");
            return false;
        }

        //Photo Config Check
        if (!$this->checkConfig()) {
            Log::error("Image size config error");
            return false;
        }

        $image = $this->image;
        $format = $this->format;

        return $this->previewImage($image, $format);
    }

}
