<?php

namespace Nette\Upload\Controllers;

use UploadManager;
use Illuminate\Http\Request;

use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;


use Image;
use Nette\Upload\Models\Media;
use Intervention\Image\ImageManager as InterventionImageManager;


/**
 * #### Handles all Media Upload Requests ####
 * @package App\Http\Controllers
 */
class MediaController extends BaseController
{

    use AuthorizesRequests, DispatchesJobs, ValidatesRequests;

    /**
     * MediaController constructor.
     */
    function __construct()
    {

    }

    #region Session Methods

    /**
     * Single File Upload
     *
     * @param Request $request
     * @return string
     */
    public function upload(Request $request)
    {
        UploadManager::set($request->skey);
        $file = ['payload' => $request->file];
        UploadManager::add($request->type, $request->key, $file, $request->templateID);
        $result = UploadManager::get($request->type, $request->key);
        $send = [];
        foreach ($result as $key => $item) {
            $send[$key] = [
                'uuid' => $item['file']['uuid'],
                'templateID' => $item['templateID'],
                'clientname' => $item['file']['clientName'],
            ];
        }
        return response()->json($send);

    }

    /**
     * Multiple File upload
     *
     * @param Request $request
     * @return string
     */
    public function multiUpload(Request $request)
    {
        UploadManager::set($request->skey);
        $allFiles = $request->allFiles();
        $files = [];
        foreach ($allFiles as $filePayload) {
            $files[] = ['payload' => $filePayload];
        }
        //Multiple File Upload Add All Files To Uploader
        UploadManager::add($request->type, $request->key, $files, null, true);
        $result = UploadManager::get($request->type, $request->key);

        $send = [];
        foreach ($result as $key => $item) {
            $send[$key] = [
                'uuid' => $item['file']['uuid'],
                'templateID' => $item['templateID']
            ];
        }
        return response()->json($send);
    }

    /**
     * File Upload from Base64 blob
     * @param Request $request
     * @return string
     */
    public function blobUpload(Request $request)
    {
        UploadManager::set($request->skey);
        $file = ['payload' => $request->payload];
        UploadManager::add($request->type, $request->key, $file, $request->templateID, false, 'base64');
        $result = UploadManager::get($request->type, $request->key);
        $send = [];
        foreach ($result as $key => $item) {
            $send[$key] = [
                'uuid' => $item['file']['uuid'],
                'templateID' => $item['templateID']
            ];
        }
        return response()->json($send);
    }

    /**
     * Get item data for incompleted uploads
     *
     * @param Request $request
     * @return string
     */
    public function get(Request $request)
    {
        UploadManager::set($request->_skey);
        return response()->json(UploadManager::get($request->type, $request->key));
    }

    /**
     * Remove Uploaded Items
     *
     * @param Request $request
     * @return string
     */
    public function remove(Request $request)
    {
        UploadManager::set($request->skey);
        return response()->json(UploadManager::delete($request->type, $request->key, $request->id));
    }

    public function removeAll(Request $request)
    {
        UploadManager::set($request->skey);
        return response()->json(UploadManager::deleteAll($request->type, $request->key));
    }

    /**
     * Get Image preview for specified uploaded items
     *
     * @param Request $request
     * @return string
     */
    public function getPreview(Request $request)
    {
//        dd($request->session());
        UploadManager::set($request->skey);
        $send = $item = UploadManager::get($request->type, $request->key, $request->id);

        if ($item) {
            $send = [
                'templateID' => $item['templateID'],
                'uuid' => $item['file']['uuid'],
                'clientname' => $item['file']['clientName'],
            ];

            if(isset($item['preview'])){
                $send['preview'] = $item['preview'];
            }

        }
        return json_encode($send);
    }

    /**
     * Generate and Get Image Placeholder on request
     *
     * @param Request $request
     * @param int $width
     * @param int $height
     * @param null $text
     * @param null $fgc
     * @param string $bgc
     * @return mixed
     */
    public function getPlaceHolder(Request $request)
    {
        $width = (int)$request->get('width', 200);
        $height = (int)$request->get('height', 200);
        $text = $request->get('text', null);

        if($width == 0){
            $width = 200;
        }

        if($height == 0){
            $height = 200;
        }
        $bgc = '333';


        $manager = new InterventionImageManager();
        $img = $manager->canvas($width, $height, '#' . $bgc);

        if($text === null) {
            $text = sprintf("%s x %s", $width, $height);
        }

        $font_size = 32;
        if ($width <= 200 && $height <= 150) {
            $font_size = 16;
        }
        if ($width >= 60 && $height >= 30) {
            $img->text($text, $width / 2, $height / 2, function ($font) use ($font_size) {
                $font->file(4);
                $font->size($font_size);
                $font->color([255, 255, 255]);
                $font->align('center');
                $font->valign('middle');

            });
        }

        return $img->response('png');
    }

    #endregion

    #region Model Methods

    /**
     * Remove Already uploaded media models and its files.
     *
     * @param Request $request
     * @param         $id
     * @return string
     */
    public function removeModel(Request $request, $id)
    {
        return deleteMedia($id);
    }
    #endregion

}
