<?php

namespace Nette\Admin\Models;

use Cocur\Slugify\Slugify;
use Illuminate\Database\Eloquent\Model;

/**
 * #### Base Model####
 *
 * Manages all Model
 *
 * @package App\Models
 */
class BaseModel extends Model
{

    public function customizeSlugEngine(Slugify $engine, $attribute)
    {
        $engine->activateRuleset('turkish');
        $engine->addRule('\'', '');
        return $engine;
    }

    public function getFillable()
    {
        $fillable = $this->fillable ?? [];
        $default = [
            'created_at'
        ];
        return array_merge($fillable, $default);
    }

    public function scopePublished($query)
    {
        return $query->where('status', true);
    }

    public function scopeActive($query)
    {
        return $query->where('status', true);
    }

    public function getModelFromTable($table)
    {
        foreach (get_declared_classes() as $class) {
            if (is_subclass_of($class, 'Illuminate\Database\Eloquent\Model')) {
                $model = new $class;
                if ($model->getTable() === $table)
                    return $class;
            }
        }

        return false;
    }

    //region Elastic Related Fields
    public function getIndexDocumentData()
    {
        $mapping = [];
        foreach ($this->elasticFields ?? [] as $field) {
            $mapping[$field] = $this->{$field};
        }

        foreach ($this->elasticRelations ?? [] as $relation) {
            $mapping[$relation] = $this->{$relation};
        }

        return $mapping;
    }

    public static function elasticFields(): array
    {
        return parent::elasticFields();
    }

    public static function elasticRelations(): array
    {
        return parent::getElasticRelations();
    }

    public static function mappingProperties(): array
    {
        return parent::getMappingProperties();
    }

    public function getElasticFields()
    {
        return $this->elasticFields;
    }

    public function getElasticRelations()
    {
        return $this->elasticRelations;
    }

    public function getMappingProperties()
    {

        $default = [
            'id' => [
                'type' => 'integer'
            ],
            'meta' => [
                'type' => 'nested',
                'properties' => [
                    'title' => [
                        'type' => 'text',
                    ],
                    'description' => [
                        'type' => 'text'
                    ]
                ]
            ],
            'created_at' => [
                'type' => 'date',
                "format" => "yyyy-MM-dd H:m:s||strict_date_optional_time||epoch_millis"
            ],
            'updated_at' => [
                'type' => 'date',
                "format" => "yyyy-MM-dd H:m:s||strict_date_optional_time||epoch_millis"
            ],
        ];
        return array_merge($this->mappingProperties, $default);
    }

    //endregion


    public function getMetaAttribute($value)
    {
        if (is_array($value)) {
            return $value;
        } else {
            return json_decode($value, true);
        }
    }

    public function getDatePublishedAttribute()
    {
        return $this->created_at ? $this->created_at->format("c") : null;
    }

    public function getDateModifiedAttribute()
    {
        return $this->updated_at ? $this->updated_at->format("c") : null;
    }

    //türkçe tarih string i döndürür.
    public function getCreatedAtStrAttribute()
    {
        return $this->created_at ? $this->created_at->formatLocalized('%d %B %Y %H:%M') : '';
    }

    public function getUpdatedAtStrAttribute()
    {
        return $this->updated_at ? $this->updated_at->formatLocalized('%d %B %Y %H:%M') : '';
    }

}
